function  [Subpop, peaks, p] = ACCENSE_cluster(y_out,X,markers)

%Performs kernel based density estimation and peak detection to find
%cellular subpopulations. Samples subpopulations and computes the
%phenotypic signatures

%Karthik Shekhar, MIT 2013 
%Requires functions kernel_density(), DetectPeaks() and xticklabel_rotate

close all

%arg-check
if nargin < 2
    flag_X = 0;
else flag_X = 1;
    if size(y_out,1) ~= size(X,1)
    disp('ERROR : y_out and X should have the same number of datapoints. Exiting. Please check input...');
    return
end
end


%Assign markers numbers if the variable does not exist
if ~exist('markers','var') && flag_X == 1
    disp('Marker names have not been provided. Using integers 1, 2, ...');
    for i=1:size(X,2)
        markers{i} = num2str(i);
    end
end


%Create Custom Colormap
w = [1 1 1]; %Start
r = [0.5 0 0]; %End
b = [0 0 0.5]; %Middle
bb = [0 1 1];
g = [0 1 0];
o = [1 1 0];

clear c1 c2 c3 c4
for i=1:3
    c1(:,i) = linspace(w(i),bb(i),5);
end

for i=1:3
    c2(:,i) = linspace(bb(i),b(i),30);
end

for i = 1:3
    c3(:,i) = linspace(b(i),g(i),20);
end

for i = 1:3
    c4(:,i) = linspace(g(i),o(i),10);
end

for i = 1:3
    c5(:,i) = linspace(o(i),r(i),10);
end


c = [c1;c2; c3; c4; c5];




%Find optimal kernel bandwidth
disp('Finding the optimal kernel bandwidth. This might take a couple of hours (consider parallelizing).. ')
y_range = min([max(y_out(:,1))-min(y_out(:,1)); max(y_out(:,2)) - min(y_out(:,2))]);

sig_min = y_range/200; sig_max = y_range/20;
sig_range = linspace(sig_min,sig_max,30);


%matlabpool 8     %Uncomment matlabpool commands if you wish to parallelize

for i=1:length(sig_range) %Change 'for' to 'parfor' if you wish to parallelize
    
    fprintf('Evaluating clusters at bandwidth = %3.1f\n',sig_range(i));
    [SNEmap, z1_range, z2_range] = kernel_density(y_out(:,1),y_out(:,2),sig_range(i));
    p = DetectPeaks(flipud(SNEmap));
    Npeaks(i) = size(p,1)/2;
            
end

%matlabpool close

%Locate plateau
for i=1:length(sig_range)
    if i > 1
        if Npeaks(i-1) - Npeaks(i) <= 1
%             sig_opt = sig_range(i-1);
            sig_opt = sig_range(i);  %Walid Modification (i) instead of (i-1)
            Nsub = Npeaks(i-1);
            break;
        end
    end
end

figure
plot(sig_range,Npeaks,'k','LineWidth',3);
set(gca,'FontSize',14);
xlabel('$\gamma$','Interpreter','Latex','FontSize',20);
ylabel('$N_{peaks}$','Interpreter','Latex','FontSize',20);
% saveas(gcf,'Npeaks_bandwidth','epsc'); hold off;


if ~exist('sig_opt','var')
    disp('Could not locate a plateau. Consider changing the kernel search parameters on line 50-51');
    reply = input('Would you like to provide a bandwidth (y/n):', 's');
    if reply == 'y'
        sig_opt = input('Please provide a value (Warning : This will directly determine # of subpopulations):');
    else
        disp('Exiting. Consider changing kernel search parameters');
        return
    end
else
    fprintf('Based on plateau in Npeaks vs. gamma, the optimal kernel bandwidth selected is %3.2f,\n and %i subpopulations are identified\n',sig_opt,Nsub);
    reply = input('Do you want to overrule this and provide your own value of the bandwidth? (y/n):', 's');

    if reply == 'y'
        sig_opt = input('Please specify the desired kernel bandwidth (Warning : This will directly determine # of subpopulations):');
    end
end

%Compute density map
[SNEmap, z1_range, z2_range] = kernel_density(y_out(:,1),y_out(:,2),sig_opt);

%Find peaks (Returns pixel coordinates)
p = DetectPeaks(flipud(SNEmap));

%Convert pixel coordinates to actual coordinates
peaks = [z1_range(p(2:2:end))' z2_range(length(z2_range) - p(1:2:end))'];

%Show 2-d density maps with subpopulation locations
figure
imagesc(flipud(SNEmap)); hold on;
plot(p(2:2:end),p(1:2:end),'ko','MarkerSize',5,'MarkerFaceColor','k')
colormap(c);
xlabel('$y_1$','Interpreter','Latex','FontSize',20);
ylabel('$y_2$','Interpreter','Latex','FontSize',20);
title('Subpopulation Centers','FontSize',20);
set(gca, 'XTick', [1 400]);
set(gca, 'XTickLabel', [min(y_out(:,1)) max(y_out(:,1))],'FontSize',12);
set(gca, 'YTick', [1 400]);
set(gca, 'YTickLabel', fliplr([min(y_out(:,2)) max(y_out(:,2))]),'FontSize',12);
colorbar
% saveas(gcf,'subpopulations_marked','epsc'); hold off;



%Number subpopulations
figure
imagesc(flipud(SNEmap)); hold on;
colormap(c)
for i=1:size(p,1)/2
    text(p(2*i,1),p(2*i-1,1),num2str(i),'FontSize',14,'FontWeight','bold','Color','k');
end
xlabel('$y_1$','Interpreter','Latex','FontSize',20);
ylabel('$y_2$','Interpreter','Latex','FontSize',20);
set(gca, 'XTick', [1 400]);
set(gca, 'XTickLabel', [min(y_out(:,1)) max(y_out(:,1))],'FontSize',12);
set(gca, 'YTick', [1 400]);
set(gca, 'YTickLabel', fliplr([min(y_out(:,2)) max(y_out(:,2))]),'FontSize',12);
colorbar
saveas(gcf,'Numbered_subpopulations','epsc'); hold off;
hold off

%If X is not present, return

if flag_X == 0
    return
end

figure
%Obtain subpopulation samples
for i=1:size(peaks,1)
    
    %Find closest subpopulation and determine appropriate radius to sample
    dists = sqrt(sum((repmat(peaks(i,:),size(peaks,1)-1,1) - [peaks(1:i-1,:); peaks(i+1:size(peaks,1),:)]).^2,2));
    
    min_dist = min(dists);
    
    ind = find(sum((y_out - repmat(peaks(i,:),size(y_out,1),1)).^2,2) < (min_dist/2)^2);
    
    Subpop{i}.X = X(ind,:);
    Subpop{i}.size = size(Subpop{i}.X,1);
    Subpop{i}.ind = ind;
    
    scatter(y_out(ind,1),y_out(ind,2),4,'filled'); hold on;
    xlabel('$y_1$','Interpreter','Latex','FontSize',20);
    ylabel('$y_2$','Interpreter','Latex','FontSize',20);
    title('Cells employed in subpopulation samples to compute phenotypic signatures');
    
end


%Phenotypic signature of each subpopulation. Saves figures into a separate
%directory PhenSig

% disp('Computing the Phenoptyic signatures of all subpopulations and storing them in a directory PhenSig\n');
% if ~exist('PhenSig','dir')
%     mkdir PhenSig
% end
% copyfile('xticklabel_rotate.m','PhenSig')
% cd PhenSig
% 
% med_all = median(X);
% 
% hold off;
% for niche = 1:length(Subpop)
%     
%     cell_data = Subpop{niche}.X;
%     bar(median(cell_data),'g');hold on;
%     xlabel('Features','FontSize',15,'FontWeight','b');
%     ylabel('Signal','FontSize',15,'FontWeight','b');
%     set(gcf,'units','inches');
%     set(gcf,'papersize',[12,8])
%     set(gcf,'PaperPosition',[0 0 10.12 6.56]);
%     errorbar([1:size(X,2)],median(cell_data),mad(cell_data,1)/2,'.k');
%     
%     %Plot median of full population
%     for j=1:length(markers)
%         plot([j-0.4 j+0.4],[med_all(j) med_all(j)],'LineWidth',2); hold on;
%     end
%     
%     title(['Subpopulation ' num2str(niche) ', N = ' num2str(size(cell_data,1))], 'FontSize',14);
%     xlim([0 size(cell_data,2)+1]);ylim([min(median(X))-0.5 max(median(X))+2]);
%     xticklabel_rotate([1:size(cell_data,2)],45,markers,'FontSize',8,'FontWeight','b');
%     set(gca,'FontSize',8,'FontWeight','b');
%     saveas(gcf,['PhenSig_' num2str(niche)],'epsc'); hold off;
%     close
% end
% cd ..


end

